package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.beanutils.PropertyUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.model.InventoryUnit;
import com.floreantpos.model.InventoryUnitGroup;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class InventoryUnitGroupDAO extends BaseInventoryUnitGroupDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public InventoryUnitGroupDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	@Override
	protected void delete(Object obj, Session session) {
		InventoryUnitGroup unitGroup = (InventoryUnitGroup) obj;
		List<InventoryUnit> units = unitGroup.getUnits();
		if (units != null && units.size() > 0) {
			int groupUnitCount = 0;
			for (InventoryUnit inventoryUnit : units) {
				if (inventoryUnit.isDeleted()) {
					continue;
				}
				groupUnitCount++;
			}
			if (groupUnitCount > 0) {
				throw new PosException(Messages.getString("InventoryUnitGroupDAO.0")); //$NON-NLS-1$
			}
		}
		super.delete(obj, session);
	}

	@Override
	public List<InventoryUnitGroup> findAll() {
		List<InventoryUnitGroup> inventoryUnitGroups = super.findAll();
		if (inventoryUnitGroups != null) {
			for (InventoryUnitGroup inventoryUnitGroup : inventoryUnitGroups) {
				List<InventoryUnit> allInventoryUnits = inventoryUnitGroup.getUnits();
				if (allInventoryUnits != null) {
					for (Iterator iterator = allInventoryUnits.iterator(); iterator.hasNext();) {
						InventoryUnit inventoryUnit = (InventoryUnit) iterator.next();
						if (inventoryUnit.isDeleted()) {
							iterator.remove();
						}
					}
				}
			}
		}
		return inventoryUnitGroups;
	}

	public InventoryUnitGroup findByName(String unitGroupName) {
		Session session = null;
		try {
			session = createNewSession();
			return findByName(unitGroupName, session);
		} finally {
			closeSession(session);
		}
	}

	public InventoryUnitGroup findByName(String unitGroupName, Session session) {
		if (session == null)
			session = getSession();

		Criteria criteria = session.createCriteria(getReferenceClass());
		criteria.add(Restrictions.eq(InventoryUnitGroup.PROP_NAME, unitGroupName));

		return (InventoryUnitGroup) criteria.uniqueResult();
	}

	public void saveOrUpdateInventoryUnitGroups(List<InventoryUnitGroup> dataList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {

		if (dataList == null)
			return;

		Transaction tx = null;
		Session session = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();
			List<String> uniqueNames = new ArrayList<>();
			for (Iterator<InventoryUnitGroup> iterator = dataList.iterator(); iterator.hasNext();) {
				InventoryUnitGroup item = (InventoryUnitGroup) iterator.next();
				if (uniqueNames.contains(item.getName())) {
					item.setName(item.getName() + "_"); //$NON-NLS-1$
				}
				uniqueNames.add(item.getName());
				InventoryUnitGroup existingItem = get(item.getId());
				if (existingItem != null) {
					if (!BaseDataServiceDao.get().shouldSave(item.getLastUpdateTime(), existingItem.getLastUpdateTime())) {
						PosLog.info(getClass(), item.getName() + " already updated"); //$NON-NLS-1$
						continue;
					}
					long version = existingItem.getVersion();
					PropertyUtils.copyProperties(existingItem, item);
					existingItem.setVersion(version);
					existingItem.setUpdateLastUpdateTime(updateLastUpdateTime);
					existingItem.setUpdateSyncTime(updateSyncTime);
					update(existingItem, session);
				}
				else {
					item.setUpdateLastUpdateTime(updateLastUpdateTime);
					item.setUpdateSyncTime(updateSyncTime);
					save(item, session);
				}
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			closeSession(session);
		}

	}

}