package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.PosLog;
import com.floreantpos.model.InventoryTransaction;
import com.floreantpos.model.InventoryVendor;
import com.floreantpos.model.InventoryVendorItems;
import com.floreantpos.model.MenuItem;
import com.floreantpos.model.PurchaseOrder;
import com.floreantpos.model.PurchaseOrderItem;
import com.floreantpos.swing.PaginatedListModel;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class InventoryVendorItemsDAO extends BaseInventoryVendorItemsDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public InventoryVendorItemsDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void delete(Object obj, Session s) {
		if (obj instanceof InventoryVendorItems) {
			InventoryVendorItems vendorItems = (InventoryVendorItems) obj;
			vendorItems.setVendor(null);
			vendorItems.setItem(null);
			super.update(vendorItems, s);
		}
		else {
			super.delete(obj, s);
		}
	}

	@SuppressWarnings("unchecked")
	@Override
	public List<InventoryVendorItems> findAll() {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(this.getReferenceClass());
			criteria.add(Restrictions.and(Restrictions.isNotNull(InventoryVendorItems.PROP_VENDOR), Restrictions.isNotNull(InventoryVendorItems.PROP_ITEM)));
			return criteria.list();
		}
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	public List<InventoryVendorItems> findByItem(MenuItem item) {
		if (item.getId() == null) {
			return null;
		}
		Session session = null;

		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			if (item != null) {
				criteria.add(Restrictions.eq(InventoryVendorItems.PROP_ITEM, item));
			}

			List list = criteria.list();
			if (list.isEmpty()) {
				return null;
			}

			return list;
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public InventoryVendorItems findByItemAndVendor(MenuItem item, InventoryVendor vendor) {
		if (item == null || StringUtils.isBlank(item.getId()) || vendor == null) {
			return null;
		}
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(InventoryVendorItems.PROP_ITEM, item));
			criteria.add(Restrictions.eq(InventoryVendorItems.PROP_VENDOR, vendor));
			criteria.setMaxResults(1);
			return (InventoryVendorItems) criteria.uniqueResult();
		}
	}

	public boolean vendorHasItem(InventoryVendor vendor, MenuItem item, Session session) {
		if (item.getId() == null) {
			return false;
		}
		boolean isNewSession = false;
		try {
			if (session == null) {
				session = createNewSession();
				isNewSession = true;
			}
			Criteria criteria = session.createCriteria(getReferenceClass());
			if (item != null && vendor != null) {
				criteria.add(Restrictions.eq(InventoryVendorItems.PROP_ITEM, item));
				criteria.add(Restrictions.eq(InventoryVendorItems.PROP_VENDOR, vendor));
			}

			List list = criteria.list();
			if (list.isEmpty()) {
				return false;
			}

			return true;
		} catch (Exception e) {
			PosLog.error(InventoryVendorDAO.class, e.getMessage(), e);
		} finally {
			if (isNewSession && session != null) {
				session.close();
			}
		}
		return false;
	}

	public void saveItems(PurchaseOrder order) {
		Session session = null;
		Transaction tx = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();
			List<PurchaseOrderItem> orderItems = order.getOrderItems();
			for (PurchaseOrderItem purchaseOrderItem : orderItems) {
				MenuItem menuItem = purchaseOrderItem.getMenuItem();
				InventoryVendor vendor = order.getVendor();
				if (!vendorHasItem(vendor, menuItem, null)) {
					InventoryVendorItems inventoryVendorItems = new InventoryVendorItems();
					inventoryVendorItems.setItem(menuItem);
					inventoryVendorItems.setVendor(vendor);
					session.saveOrUpdate(inventoryVendorItems);
				}
			}

			session.saveOrUpdate(order);
			tx.commit();
		} catch (Exception e) {
			if (tx != null) {
				tx.rollback();
			}
			throwException(e);
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public void saveItems(InventoryTransaction transaction, Session session) {
		try {
			MenuItem menuItem = transaction.getMenuItem();
			InventoryVendor vendor = null;
			if (transaction.getVendorId() != null) {
				vendor = InventoryVendorDAO.getInstance().get(transaction.getVendorId());
			}
			if (!vendorHasItem(vendor, menuItem, session)) {
				InventoryVendorItems inventoryVendorItems = new InventoryVendorItems();
				inventoryVendorItems.setItem(menuItem);
				inventoryVendorItems.setVendor(vendor);
				session.saveOrUpdate(inventoryVendorItems);
			}
		} catch (Exception e) {
			PosLog.error(InventoryVendorDAO.class, e.getMessage(), e);
		}
	}

	public void saveAll(List<InventoryVendorItems> vendorItemsList) {
		Transaction tx = null;
		Session session = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();
			saveAll(vendorItemsList, session);
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw new RuntimeException(e);
		} finally {
			closeSession(session);
		}
	}

	public void saveAll(List<InventoryVendorItems> vendorItemsList, Session session) {
		for (InventoryVendorItems inventoryVendorItems : vendorItemsList) {
			saveOrUpdate(inventoryVendorItems, session);
		}
	}

	public List<InventoryVendorItems> findByVendor(InventoryVendor vendor) {
		if (vendor.getId() == null) {
			return null;
		}
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			if (vendor != null) {
				criteria.add(Restrictions.eq(InventoryVendorItems.PROP_VENDOR, vendor));
			}

			List list = criteria.list();
			if (list.isEmpty()) {
				return null;
			}
			return list;
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public void loadVendorItems(PaginatedListModel model, InventoryVendor vendor) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(getReferenceClass());
			if (vendor != null) {
				criteria.add(Restrictions.eq(InventoryVendorItems.PROP_VENDOR, vendor));
			}
			criteria.setProjection(Projections.rowCount());
			Number rowCount = (Number) criteria.uniqueResult();
			model.setNumRows(rowCount.intValue());
			criteria.setProjection(null);

			criteria.setFirstResult(model.getCurrentRowIndex());
			criteria.setMaxResults(model.getPageSize());
			List list = criteria.list();
			for (Object object : list) {
				if (object instanceof InventoryVendorItems) {
					InventoryVendorItems vendorItems = (InventoryVendorItems) object;
					if (vendorItems.getItem() != null && vendorItems.getItem().getId() != null) {
						model.setData(criteria.list());
					}
				}
			}

		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public void saveOrUpdateInventoryVendorItems(List<InventoryVendorItems> dataList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {

		if (dataList == null)
			return;

		Transaction tx = null;
		Session session = null;
		Map<String, MenuItem> itemMap = new HashMap<>();
		try {
			session = createNewSession();
			tx = session.beginTransaction();

			for (Iterator<InventoryVendorItems> iterator = dataList.iterator(); iterator.hasNext();) {
				InventoryVendorItems item = (InventoryVendorItems) iterator.next();
				PosLog.debug(getClass(), "Inventory vendor item id: " + item.getId()); //$NON-NLS-1$
				
				MenuItem existingMenuItem = item.getItem();
				if (existingMenuItem != null) {
					MenuItem mapItem = itemMap.get(existingMenuItem.getId());
					if (mapItem != null) {
						item.setItem(mapItem);
					}
					else {
						existingMenuItem = MenuItemDAO.getInstance().get(existingMenuItem.getId());
						item.setItem(existingMenuItem);
						itemMap.put(existingMenuItem.getId(), existingMenuItem);
					}
				}
				InventoryVendorItems existingItem = get(item.getId());
				if (existingItem == null) {
					existingItem = findByItemAndVendor(item.getItem(), item.getVendor());
				}
				if (existingItem != null) {
					if (!BaseDataServiceDao.get().shouldSave(item.getLastUpdateTime(), existingItem.getLastUpdateTime())) {
						PosLog.info(getClass(), item.getId() + " already updated"); //$NON-NLS-1$
						continue;
					}
					long version = existingItem.getVersion();
					String id = existingItem.getId();
					PropertyUtils.copyProperties(existingItem, item);
					existingItem.setVersion(version);
					existingItem.setId(id);
					existingItem.setUpdateLastUpdateTime(updateLastUpdateTime);
					existingItem.setUpdateSyncTime(updateSyncTime);
					update(existingItem, session);
				}
				else {
					item.setUpdateLastUpdateTime(updateLastUpdateTime);
					item.setUpdateSyncTime(updateSyncTime);
					save(item, session);
				}
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			closeSession(session);
		}
	}

	public void nullifyByInventoryVendor(InventoryVendor inventoryVendor, Session session) {
		try {
			String hql = "UPDATE %s items SET items.%s = :vendor1, items.%s = :item WHERE items.%s = :vendor0"; //$NON-NLS-1$
			hql = String.format(hql, InventoryVendorItems.REF, InventoryVendorItems.PROP_VENDOR, InventoryVendorItems.PROP_ITEM,
					InventoryVendorItems.PROP_VENDOR);
			Query query = session.createQuery(hql);
			query.setParameter("vendor1", null); //$NON-NLS-1$
			query.setParameter("item", null); //$NON-NLS-1$
			query.setParameter("vendor0", inventoryVendor); //$NON-NLS-1$
			query.executeUpdate();
		} catch (Exception e) {
			throw e;
		}
	}

	public void saveOrUpdateMenuItemVendors(MenuItem menuItem, List<InventoryVendorItems> vendorItems) {
		if (menuItem == null) {
			return;
		}
		Transaction tx = null;
		try (Session session = createNewSession()) {
			tx = session.beginTransaction();
			List<String> ids = new ArrayList<>();
			if (vendorItems != null && vendorItems.size() > 0) {
				for (InventoryVendorItems inventoryVendorItems : vendorItems) {
					inventoryVendorItems.setItem(menuItem);
					saveOrUpdate(inventoryVendorItems, session);
					ids.add(inventoryVendorItems.getId());
				}
			}
			clearVendorItems(menuItem, session, ids);
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		}
	}

	private void clearVendorItems(MenuItem menuItem, Session session, List<String> ids) {
		if (ids.size() == 0) {
			String hqlString = "delete from " + InventoryVendorItems.REF + " where %s=:item"; //$NON-NLS-1$ //$NON-NLS-2$
			Query query = session.createQuery(String.format(hqlString, InventoryVendorItems.PROP_ITEM));
			query.setParameter("item", menuItem); //$NON-NLS-1$
			query.executeUpdate();
		}
		else {
			String hqlString = "delete from " + InventoryVendorItems.REF + " where %s=:item and %s not in (:ids)"; //$NON-NLS-1$ //$NON-NLS-2$
			Query query = session.createQuery(String.format(hqlString, InventoryVendorItems.PROP_ITEM, InventoryVendorItems.PROP_ID));
			query.setParameter("item", menuItem); //$NON-NLS-1$
			query.setParameterList("ids", ids); //$NON-NLS-1$
			query.executeUpdate();
		}
	}

	public void saveOrUpdateVendorMenuItems(InventoryVendor vendor, List<InventoryVendorItems> addedVendorItems,
			List<InventoryVendorItems> removedVendorItems) {
		if (vendor == null) {
			return;
		}
		Transaction tx = null;
		try (Session session = createNewSession()) {
			tx = session.beginTransaction();
			List<String> addedIds = new ArrayList<>();
			if (addedVendorItems != null && addedVendorItems.size() > 0) {
				for (InventoryVendorItems inventoryVendorItems : addedVendorItems) {
					inventoryVendorItems.setVendor(vendor);
					saveOrUpdate(inventoryVendorItems, session);
					addedIds.add(inventoryVendorItems.getId());
				}
			}
			if (removedVendorItems != null && removedVendorItems.size() > 0) {
				List<String> toBeRemoveIds = new ArrayList<>();
				for (InventoryVendorItems inventoryVendorItems : removedVendorItems) {
					if (addedIds.contains(inventoryVendorItems.getId())) {
						continue;
					}
					toBeRemoveIds.add(inventoryVendorItems.getId());
				}
				if (toBeRemoveIds.size() > 0) {
					removeVendorItems(vendor, session, toBeRemoveIds);
				}
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		}
	}

	private void removeVendorItems(InventoryVendor vendor, Session session, List<String> ids) {
		String hqlString = "delete from " + InventoryVendorItems.REF + " where %s=:vendor and %s in (:ids)"; //$NON-NLS-1$ //$NON-NLS-2$
		Query query = session.createQuery(String.format(hqlString, InventoryVendorItems.PROP_VENDOR, InventoryVendorItems.PROP_ID));
		query.setParameter("vendor", vendor); //$NON-NLS-1$
		query.setParameterList("ids", ids); //$NON-NLS-1$
		query.executeUpdate();
	}
}