package com.floreantpos.model.dao;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.model.Customer;
import com.floreantpos.model.OrderHistory;
import com.floreantpos.model.Pagination;
import com.floreantpos.model.util.DateUtil;
import com.floreantpos.swing.PaginationSupport;
import com.orocube.common.util.TicketStatus;

public class OrderHistoryDAO extends BaseOrderHistoryDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public OrderHistoryDAO() {
	}

	@SuppressWarnings("unchecked")
	public List<OrderHistory> findOrdersByCustomerId(String customerId) {
		if (StringUtils.isBlank(customerId)) {
			return new ArrayList<>(0);
		}

		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(OrderHistory.class);
			criteria.add(Restrictions.eq(OrderHistory.PROP_CUSTOMER_ID, customerId));
			criteria.add(Restrictions.isNotNull(OrderHistory.PROP_TICKET_JSON));
			criteria.addOrder(Order.desc(OrderHistory.PROP_ORDER_DATE));
			criteria.setMaxResults(50);

			return criteria.list();
		}
	}

	public OrderHistory findOrder(String orderId, String customerId) {
		if (StringUtils.isBlank(orderId) || StringUtils.isBlank(customerId)) {
			return null;
		}
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(OrderHistory.class);
			criteria.add(Restrictions.eq(OrderHistory.PROP_ID, orderId));
			criteria.add(Restrictions.eq(OrderHistory.PROP_CUSTOMER_ID, customerId));
			criteria.add(Restrictions.isNotNull(OrderHistory.PROP_TICKET_JSON));
			criteria.addOrder(Order.desc(OrderHistory.PROP_ORDER_DATE));
			return (OrderHistory) criteria.uniqueResult();
		}
	}

	public void loadOrdersByCustomerId(PaginationSupport tableModel, String customerId) {
		if (StringUtils.isBlank(customerId)) {
			tableModel.setNumRows(0);
			tableModel.setRows(new ArrayList<>());
			return;
		}
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(OrderHistory.class);
			criteria.setProjection(Projections.rowCount());
			criteria.add(Restrictions.eq(OrderHistory.PROP_CUSTOMER_ID, customerId));
			criteria.add(Restrictions.isNotNull(OrderHistory.PROP_TICKET_JSON));
			Number result = (Number) criteria.uniqueResult();
			int rowCount = result == null ? 0 : result.intValue();
			tableModel.setNumRows(rowCount);
			if (rowCount == 0) {
				tableModel.setRows(new ArrayList<>());
				return;
			}
			criteria.setProjection(null);
			criteria.addOrder(Order.desc(OrderHistory.PROP_ORDER_DATE));
			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());
			tableModel.setRows(criteria.list());
		}
	}

	public void loadData(Pagination<OrderHistory> tableModel, String invoiceId, String customerEmail, String status) {
		Customer customer = null;
		if (StringUtils.isNotBlank(customerEmail)) {
			customer = CustomerDAO.getInstance().findByEmail(customerEmail);
			if (customer == null) {
				tableModel.setNumRows(0);
				tableModel.setRows(new ArrayList<>());
				return;
			}
		}
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(OrderHistory.class);
			criteria.setProjection(Projections.rowCount());
			criteria.add(Restrictions.isNotNull(OrderHistory.PROP_TICKET_JSON));
			if (StringUtils.isNotBlank(invoiceId)) {
				criteria.add(Restrictions.eq(OrderHistory.PROP_ID, invoiceId.trim()).ignoreCase());
			}
			criteria.add(Restrictions.eq(OrderHistory.PROP_ORDER_TYPE, "invoice").ignoreCase()); //$NON-NLS-1$  
			if (customer != null) {
				criteria.add(Restrictions.eq(OrderHistory.PROP_CUSTOMER_ID, customer.getId()));
			}
			if (StringUtils.isNotBlank(status)) {
				if (status.equalsIgnoreCase(OrderHistory.STATUS_EXPIRED)) {
					criteria.add(Restrictions.and(Restrictions.lt(OrderHistory.PROP_EXPIRY_DATE, DateUtil.startOfDay(StoreDAO.getServerTimestamp())),
							Restrictions.eq(OrderHistory.PROP_ORDER_STATUS, TicketStatus.Pending.getDisplayString()).ignoreCase()));
				}
				else if (status.equalsIgnoreCase(TicketStatus.Pending.getDisplayString())) {
					criteria.add(Restrictions.and(Restrictions.ge(OrderHistory.PROP_EXPIRY_DATE, DateUtil.startOfDay(StoreDAO.getServerTimestamp())),
							Restrictions.eq(OrderHistory.PROP_ORDER_STATUS, TicketStatus.Pending.getDisplayString()).ignoreCase()));
				}
				else {
					criteria.add(Restrictions.eq(OrderHistory.PROP_ORDER_STATUS, status).ignoreCase()); //$NON-NLS-1$
				}
			}
			Number result = (Number) criteria.uniqueResult();
			int rowCount = result == null ? 0 : result.intValue();
			tableModel.setNumRows(rowCount);
			if (rowCount == 0) {
				tableModel.setRows(new ArrayList<>());
				return;
			}
			criteria.setProjection(null);
			criteria.addOrder(Order.desc(OrderHistory.PROP_ORDER_DATE));
			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());
			tableModel.setRows(criteria.list());
		}
	}
}