package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.criterion.Conjunction;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;
import org.hibernate.transform.Transformers;

import com.floreantpos.DuplicateDataException;
import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.model.CustomerGroup;
import com.floreantpos.model.Department;
import com.floreantpos.model.MenuItem;
import com.floreantpos.model.OrderType;
import com.floreantpos.model.PriceRule;
import com.floreantpos.model.PriceShift;
import com.floreantpos.model.PriceTable;
import com.floreantpos.model.PriceTableItem;
import com.floreantpos.model.SalesArea;
import com.floreantpos.util.ShiftUtil;

public class PriceRuleDAO extends BasePriceRuleDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public PriceRuleDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session session) {
		validateData((PriceRule) obj, session);
		updateTime(obj);
		return super.save(obj, session);
	}

	@Override
	protected void update(Object obj, Session session) {
		validateData((PriceRule) obj, session);
		updateTime(obj);
		super.update(obj, session);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session session) {
		validateData((PriceRule) obj, session);
		updateTime(obj);
		super.saveOrUpdate(obj, session);
	}

	private void validateData(PriceRule priceRule, Session session) {
		if (priceRuleExists(priceRule, session)) {
			throw new DuplicateDataException(Messages.getString("PriceRuleDAO.1")); //$NON-NLS-1$
		}
	}

	private boolean priceRuleExists(PriceRule priceRule, Session session) {
		Criteria criteria = session.createCriteria(getReferenceClass());
		String existingId = priceRule.getId();
		if (StringUtils.isNotEmpty(existingId)) {
			Criterion existingIdExp = Restrictions.ne("id", existingId); //$NON-NLS-1$
			criteria.add(existingIdExp);
		}
		addDeletedFilter(criteria);

		String outletId = priceRule.getOutletId();
		Criterion outletExp = outletId != null ? Restrictions.eq(PriceRule.PROP_OUTLET_ID, outletId) : Restrictions.isNull(PriceRule.PROP_OUTLET_ID);

		String departmentId = priceRule.getDepartmentId();
		Criterion departmentExp = departmentId != null ? Restrictions.eq(PriceRule.PROP_DEPARTMENT_ID, departmentId)
				: Restrictions.isNull(PriceRule.PROP_DEPARTMENT_ID);

		String salesAreaId = priceRule.getSalesAreaId();
		Criterion salesAreaExp = salesAreaId != null ? Restrictions.eq(PriceRule.PROP_SALES_AREA_ID, salesAreaId)
				: Restrictions.isNull(PriceRule.PROP_SALES_AREA_ID);

		String orderTypeId = priceRule.getOrderTypeId();
		Criterion orderTypeExp = orderTypeId != null ? Restrictions.eq(PriceRule.PROP_ORDER_TYPE_ID, orderTypeId)
				: Restrictions.isNull(PriceRule.PROP_ORDER_TYPE_ID);

		String customerGroupId = priceRule.getCustomerGroupId();
		Criterion customerGroupExp = customerGroupId != null ? Restrictions.eq(PriceRule.PROP_CUSTOMER_GROUP_ID, customerGroupId)
				: Restrictions.isNull(PriceRule.PROP_CUSTOMER_GROUP_ID);

		String priceShiftId = priceRule.getPriceShiftId();
		Criterion priceShiftExp = priceShiftId != null ? Restrictions.eq(PriceRule.PROP_PRICE_SHIFT_ID, priceShiftId)
				: Restrictions.isNull(PriceRule.PROP_PRICE_SHIFT_ID);

		String priceTableId = priceRule.getPriceTableId();
		Criterion priceTableExp = priceTableId != null ? Restrictions.eq(PriceRule.PROP_PRICE_TABLE_ID, priceTableId)
				: Restrictions.isNull(PriceRule.PROP_PRICE_TABLE_ID);

		Conjunction priceRuleExp = Restrictions.and(outletExp, departmentExp, salesAreaExp, orderTypeExp, customerGroupExp, priceShiftExp, priceTableExp);
		criteria.add(priceRuleExp);

		List list = criteria.list();
		if (list != null && list.size() > 0) {
			return true;
		}
		return false;
	}

	@Override
	public List<PriceRule> findAll() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			return criteria.list();
		}
	}

	@Override
	protected void delete(Object obj, Session s) {
		if (obj instanceof PriceRule) {
			PriceRule priceRule = (PriceRule) obj;
			priceRule.setDeleted(Boolean.TRUE);
			super.update(priceRule, s);
		}
		else {
			throw new PosException(Messages.getString("PriceRuleDAO.0")); //$NON-NLS-1$
		}
	}

	public Double getPrice(MenuItem menuItem, OrderType orderType, Department department, SalesArea salesArea, CustomerGroup customerGroup) {
		Session session = null;
		try {
			session = createNewSession();
			return getPrice(menuItem, orderType, department, salesArea, customerGroup, session);
		} finally {
			if (session != null) {
				try {
					session.close();
				} catch (Exception e) {
				}
			}
		}
	}

	public Double getPrice(MenuItem menuItem, OrderType orderType, Department department, SalesArea salesArea, CustomerGroup customerGroup, Session session) {
		List<String> filteredPriceShift = getFilteredPriceShift();
		Criteria criteria = session.createCriteria(getReferenceClass());
		addDeletedFilter(criteria);
		criteria.setProjection(Projections.property(PriceRule.PROP_PRICE_TABLE_ID));
		criteria.add(Restrictions.or(Restrictions.isNull(PriceRule.PROP_ORDER_TYPE_ID),
				Restrictions.eq(PriceRule.PROP_ORDER_TYPE_ID, orderType == null ? null : orderType.getId())));
		criteria.add(Restrictions.or(Restrictions.isNull(PriceRule.PROP_DEPARTMENT_ID),
				Restrictions.eq(PriceRule.PROP_DEPARTMENT_ID, department == null ? null : department.getId())));
		criteria.add(Restrictions.or(Restrictions.isNull(PriceRule.PROP_SALES_AREA_ID),
				Restrictions.eq(PriceRule.PROP_SALES_AREA_ID, salesArea == null ? null : salesArea.getId())));
		criteria.add(Restrictions.or(Restrictions.isNull(PriceRule.PROP_CUSTOMER_GROUP_ID),
				Restrictions.eq(PriceRule.PROP_CUSTOMER_GROUP_ID, customerGroup == null ? null : customerGroup.getId())));

		if (!filteredPriceShift.isEmpty()) {
			criteria.add(
					Restrictions.or(Restrictions.isNull(PriceRule.PROP_PRICE_SHIFT_ID), Restrictions.in(PriceRule.PROP_PRICE_SHIFT_ID, filteredPriceShift)));
		}
		else {
			criteria.add(Restrictions.isNull(PriceRule.PROP_PRICE_SHIFT_ID));
		}
		criteria.add(Restrictions.eq(PriceRule.PROP_ACTIVE, Boolean.TRUE));

		List list = criteria.list();
		if (list == null || list.isEmpty())
			return null;

		criteria = session.createCriteria(PriceTableItem.class);
		criteria.setProjection(Projections.property(PriceTableItem.PROP_PRICE));
		criteria.add(Restrictions.eq(PriceTableItem.PROP_MENU_ITEM_ID, menuItem.getId()));
		criteria.add(Restrictions.in(PriceTableItem.PROP_PRICE_TABLE_ID, list));

		List prices = criteria.list();
		if (prices != null && prices.size() > 0) {
			if (prices.size() == 1)
				return (Double) prices.get(0);
			else {
				return (Double) Collections.min(prices);
			}
		}
		return null;
	}

	private List<String> getFilteredPriceShift() {
		int firstPriorityValue = 1;
		List<String> filteredPriceShift = new ArrayList<>();
		List<PriceShift> priceShifts = ShiftUtil.getCurrentPricShifts();
		if (priceShifts != null) {
			for (PriceShift priceShift : priceShifts) {
				Integer priorityValue = priceShift.getPriority();
				if (firstPriorityValue >= priorityValue) {
					firstPriorityValue = priorityValue;
					filteredPriceShift.add(priceShift.getId());
				}
			}
		}
		return filteredPriceShift;
	}

	public List<PriceRule> findByPriceShiftId(String shiftId, Session session) {
		return this.findByPriceShiftId(shiftId, session, Boolean.FALSE);
	}

	public List<PriceRule> findByPriceShiftId(String shiftId, Session session, boolean addDeletedFilter) {
		if (StringUtils.isBlank(shiftId)) {
			return null;
		}

		Criteria criteria = session.createCriteria(getReferenceClass());
		if (addDeletedFilter) {
			this.addDeletedFilter(criteria);
		}
		criteria.add(Restrictions.eq(PriceRule.PROP_PRICE_SHIFT_ID, shiftId));
		criteria.setProjection(Projections.alias(Projections.property(PriceRule.PROP_NAME), PriceRule.PROP_NAME));
		criteria.setResultTransformer(Transformers.aliasToBean(PriceRule.class));
		return criteria.list();
	}

	@SuppressWarnings("unchecked")
	public List<PriceRule> getPriceRulesByPriceTable(PriceTable priceTable, Session session) {
		if (priceTable == null || priceTable.getId() == null) {
			return null;
		}
		Criteria criteria = session.createCriteria(this.getReferenceClass());
		this.addDeletedFilter(criteria);
		criteria.add(Restrictions.eq(PriceRule.PROP_PRICE_TABLE_ID, priceTable.getId()));
		criteria.setProjection(Projections.alias(Projections.property(PriceRule.PROP_NAME), PriceRule.PROP_NAME));
		return criteria.setResultTransformer(Transformers.aliasToBean(this.getReferenceClass())).list();
	}

	public void saveOrUpdatePriceRuleList(List<PriceRule> dataList, String clientOutletId, boolean updateLastUpdateTime, boolean updateSyncTime)
			throws Exception {
		if (dataList != null && dataList.size() > 0) {
			for (Iterator<PriceRule> iterator = dataList.iterator(); iterator.hasNext();) {
				PriceRule item = (PriceRule) iterator.next();
				PriceRuleDAO dao = PriceRuleDAO.getInstance();
				PriceRule existingItem = dao.get(item.getId());
				if (existingItem != null) {
					final String id = existingItem.getId();
					long version = existingItem.getVersion();
					PropertyUtils.copyProperties(existingItem, item);
					existingItem.setId(id);
					existingItem.setVersion(version);
					existingItem.setUpdateLastUpdateTime(updateLastUpdateTime);
					existingItem.setUpdateSyncTime(updateSyncTime);
					dao.update(existingItem);
				}
				else {
					item.setVersion(0);
					item.setUpdateLastUpdateTime(updateLastUpdateTime);
					item.setUpdateSyncTime(updateSyncTime);
					dao.save(item);
				}
			}
		}
	}

}