package com.floreantpos.services.report;

import java.util.Iterator;
import java.util.List;

import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.criterion.ProjectionList;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.model.CashDrawer;
import com.floreantpos.model.CashDropTransaction;
import com.floreantpos.model.CashTransaction;
import com.floreantpos.model.CreditCardTransaction;
import com.floreantpos.model.CustomPaymentTransaction;
import com.floreantpos.model.CustomerAccountTransaction;
import com.floreantpos.model.DebitCardTransaction;
import com.floreantpos.model.DeclaredTips;
import com.floreantpos.model.GiftCertificateTransaction;
import com.floreantpos.model.GratuityPaymentHistory;
import com.floreantpos.model.PayOutTransaction;
import com.floreantpos.model.PaymentType;
import com.floreantpos.model.PosTransaction;
import com.floreantpos.model.RefundTransaction;
import com.floreantpos.model.StoreSession;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.TransactionType;
import com.floreantpos.model.dao.GenericDAO;
import com.floreantpos.util.POSUtil;

public class StoreSessionSummaryReportService {

	private StoreSession storeSession;
	private Session session;
	//	private double tolarenceAmount;
	private CashDrawer sessionCashDrawer = new CashDrawer();

	public StoreSessionSummaryReportService(StoreSession storeSession) {
		super();
		this.storeSession = storeSession;
	}

	public CashDrawer populateStoreSessionSummaryReport() {
		GenericDAO dao = new GenericDAO();
		try {
			session = dao.createNewSession();

			calculateRefundAmount();
			calculateNetSales();
			//calculateDiscount();
			calculateCashTips();
			calculateChargedTips();

			double totalTips = sessionCashDrawer.getCashTips() + sessionCashDrawer.getChargedTips();
			//			sessionCashDrawer.setGratuityAmount((totalTips));

			double totalRev = sessionCashDrawer.getNetSales() /*- sessionCashDrawer.getRefundAmount()*/ - sessionCashDrawer.getTotalDiscountAmount()
					+ sessionCashDrawer.getSalesTax() + sessionCashDrawer.getServiceCharge();
			sessionCashDrawer.setTotalRevenue((totalRev));
			sessionCashDrawer.setGrossReceipts((sessionCashDrawer.getTotalRevenue() + totalTips));

			calculateCashReceipt();
			calculateCreditReceipt();
			calculateDebitReceipt();
			calculateMemberPayment();
			calculateCustomPaymentWithoutPromotion();
			calculatePromotionPayment();
			calculateGiftCertReceipts();

			calculateVoidAmount();
			calculateTipsPaid();
			calculateCashPayout();
			calculateCashBack();
			calculateDrawerBleed();
			calculateToleranceAmount();

			double totalReceiptsDiff = sessionCashDrawer.getCashReceiptAmount() + sessionCashDrawer.getCreditCardReceiptAmount()
					+ sessionCashDrawer.getDebitCardReceiptAmount() + sessionCashDrawer.getCustomerPaymentAmount() + sessionCashDrawer.getCustomPaymentAmount()
					+ sessionCashDrawer.getPromotionAmount() + sessionCashDrawer.getGiftCertChangeAmount() - sessionCashDrawer.getRefundAmount()
					- sessionCashDrawer.getToleranceAmount();

			sessionCashDrawer.setReceiptDifferential((sessionCashDrawer.getGrossReceipts() - totalReceiptsDiff));

			sessionCashDrawer.setAssignedUser(storeSession.getOpenedBy());
			sessionCashDrawer.setStartTime(storeSession.getOpenTime());
			sessionCashDrawer.setAssignedBy(storeSession.getOpenedBy());
			sessionCashDrawer.setReportTime(storeSession.getCloseTime());
			sessionCashDrawer.setClosedBy(storeSession.getClosedBy());
			sessionCashDrawer.setStoreOperationData(storeSession);

			calculateTipsDiff();
			calculateDeclaredTips();
			calculateCashDrawerBeginAmount();
			calculateDrawerAccountable();

			return sessionCashDrawer;

		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	private Criteria getCriteriaForTransaction(Class transactionClass) {
		Criteria criteria = session.createCriteria(transactionClass);
		if (transactionClass.equals(RefundTransaction.class) || transactionClass.equals(PayOutTransaction.class))
			criteria.add(Restrictions.eq(PosTransaction.PROP_TRANSACTION_TYPE, TransactionType.DEBIT.name()));
		else
			criteria.add(Restrictions.eq(PosTransaction.PROP_TRANSACTION_TYPE, TransactionType.CREDIT.name()));

		criteria.add(Restrictions.eq(PosTransaction.PROP_STORE_SESSION_ID, storeSession.getId()));
		criteria.add(Restrictions.eq(PosTransaction.PROP_VOIDED, Boolean.FALSE));
		return criteria;
	}

	private void calculateNetSales() {

		Criteria criteria = session.createCriteria(Ticket.class);
		criteria.add(Restrictions.eq(Ticket.PROP_STORE_SESSION_ID, storeSession.getId()));
		criteria.add(Restrictions.ne(Ticket.PROP_VOIDED, Boolean.TRUE));

		ProjectionList projectionList = Projections.projectionList();
		projectionList.add(Projections.sum(Ticket.PROP_SUBTOTAL_AMOUNT));
		projectionList.add(Projections.sum(Ticket.PROP_TAX_AMOUNT));
		projectionList.add(Projections.sum(Ticket.PROP_SERVICE_CHARGE));
		projectionList.add(Projections.sum(Ticket.PROP_DISCOUNT_AMOUNT));

		criteria.setProjection(projectionList);

		List<Object[]> list = criteria.list();

		for (Iterator iterator = list.iterator(); iterator.hasNext();) {
			Object[] objects = (Object[]) iterator.next();
			double salesAmnt = (objects[0] != null) ? ((Number) objects[0]).doubleValue() : 0;
			double taxAmnt = (objects[1] != null) ? ((Number) objects[1]).doubleValue() : 0;
			double serviceChrgAmnt = (objects[2] != null) ? ((Number) objects[2]).doubleValue() : 0;
			double discountAmount = (objects[3] != null) ? ((Number) objects[3]).doubleValue() : 0;

			sessionCashDrawer.setNetSales((salesAmnt));
			sessionCashDrawer.setSalesTax((taxAmnt));
			sessionCashDrawer.setServiceCharge((serviceChrgAmnt));
			//			sessionCashDrawer.setTotalDiscountSales((discountAmount));
			sessionCashDrawer.setTotalDiscountAmount((discountAmount));
		}

	}

	/*	private void calculateDiscount() {
			Criteria criteria = getCriteriaForTransaction(PosTransaction.class);
			criteria.add(Restrictions.eq(PosTransaction.PROP_STORE_SESSION_ID, storeSession.getId()));
	
			criteria.createAlias("ticket", "ticket");
			ProjectionList projectionList = Projections.projectionList();
			projectionList.add(Projections.sum("ticket." + Ticket.PROP_DISCOUNT_AMOUNT));
			criteria.setProjection(projectionList);
			sessionCashDrawer.setTotalDiscountAmount((criteria.uniqueResult()));
		}*/

	private void calculateCashTips() {
		Criteria criteria = getCriteriaForTransaction(CashTransaction.class);
		criteria.add(Restrictions.eq(PosTransaction.PROP_PAYMENT_TYPE_STRING, PaymentType.CASH.name()));
		criteria.setProjection(Projections.sum(PosTransaction.PROP_TIPS_AMOUNT));
		sessionCashDrawer.setCashTips(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	private void calculateChargedTips() {
		Criteria criteria = getCriteriaForTransaction(PosTransaction.class);
		criteria.add(Restrictions.ne(PosTransaction.PROP_PAYMENT_TYPE_STRING, PaymentType.CASH.name()));
		criteria.setProjection(Projections.sum(PosTransaction.PROP_TIPS_AMOUNT));
		sessionCashDrawer.setChargedTips(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	private void calculateCashReceipt() {
		Criteria criteria = getCriteriaForTransaction(CashTransaction.class);
		criteria.setProjection(Projections.sum(PosTransaction.PROP_AMOUNT));
		sessionCashDrawer.setCashReceiptAmount(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	private void calculateCreditReceipt() {
		Criteria criteria = getCriteriaForTransaction(CreditCardTransaction.class);
		criteria.setProjection(Projections.sum(PosTransaction.PROP_AMOUNT));

		sessionCashDrawer.setCreditCardReceiptAmount(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	private void calculateDebitReceipt() {
		Criteria criteria = getCriteriaForTransaction(DebitCardTransaction.class);
		criteria.setProjection(Projections.sum(CashTransaction.PROP_AMOUNT));
		sessionCashDrawer.setDebitCardReceiptAmount(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	//	private void calculateCustomPayment() {
	//		Criteria criteria = getCriteriaForTransaction(CustomPaymentTransaction.class);
	//		criteria.setProjection(Projections.sum(CustomPaymentTransaction.PROP_AMOUNT));
	//		sessionCashDrawer.setCustomPaymentAmount((criteria.uniqueResult()));
	//	}

	private void calculateCustomPaymentWithoutPromotion() {
		Criteria criteria = getCriteriaForTransaction(CustomPaymentTransaction.class);
		criteria.add(Restrictions.ne(PosTransaction.PROP_PAYMENT_TYPE_STRING, PaymentType.PROMOTION.name()));
		criteria.setProjection(Projections.sum(CustomPaymentTransaction.PROP_AMOUNT));
		sessionCashDrawer.setCustomPaymentAmount(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	private void calculatePromotionPayment() {
		Criteria criteria = getCriteriaForTransaction(CustomPaymentTransaction.class);
		criteria.add(Restrictions.eq(PosTransaction.PROP_PAYMENT_TYPE_STRING, PaymentType.PROMOTION.name()));
		criteria.setProjection(Projections.sum(CustomPaymentTransaction.PROP_AMOUNT));
		sessionCashDrawer.setPromotionAmount(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	private void calculateGiftCertReceipts() {
		Criteria criteria = getCriteriaForTransaction(GiftCertificateTransaction.class);
		criteria.setProjection(Projections.sum(GiftCertificateTransaction.PROP_AMOUNT));
		sessionCashDrawer.setGiftCertChangeAmount(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	private void calculateRefundAmount() {
		Criteria criteria = getCriteriaForTransaction(RefundTransaction.class);
		criteria.setProjection(Projections.sum(RefundTransaction.PROP_AMOUNT));
		sessionCashDrawer.setRefundAmount(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	private void calculateVoidAmount() {
		Criteria criteria = session.createCriteria(PosTransaction.class);
		criteria.add(Restrictions.eq(PosTransaction.PROP_TRANSACTION_TYPE, TransactionType.CREDIT.name()));
		criteria.add(Restrictions.eq(PosTransaction.PROP_STORE_SESSION_ID, storeSession.getId()));
		criteria.add(Restrictions.eq(PosTransaction.PROP_VOIDED, Boolean.TRUE));
		criteria.setProjection(Projections.sum(RefundTransaction.PROP_AMOUNT));
		sessionCashDrawer.setCardVoidAmount(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	private void calculateCashDrawerBeginAmount() {
		Criteria criteria = session.createCriteria(CashDrawer.class);
		criteria.add(Restrictions.eq(CashDrawer.PROP_STORE_SESSION_ID, storeSession.getId()));
		criteria.setProjection(Projections.sum(CashDrawer.PROP_BEGIN_CASH));
		sessionCashDrawer.setBeginCash(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	private void calculateTipsPaid() {
		Criteria criteria = session.createCriteria(GratuityPaymentHistory.class);
		criteria.add(Restrictions.eq(GratuityPaymentHistory.PROP_STORE_SESSION_ID, storeSession.getId()));
		criteria.setProjection(Projections.sum(GratuityPaymentHistory.PROP_AMOUNT));

		sessionCashDrawer.setTipsPaid(POSUtil.getDoubleAmount(criteria.uniqueResult()));

	}

	private void calculateCashPayout() {
		Criteria criteria = getCriteriaForTransaction(PayOutTransaction.class);
		criteria.setProjection(Projections.sum(PayOutTransaction.PROP_AMOUNT));
		sessionCashDrawer.setPayOutAmount(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	private void calculateDrawerBleed() {
		Criteria criteria = getCriteriaForTransaction(CashDropTransaction.class);
		criteria.setProjection(Projections.sum(CashDropTransaction.PROP_AMOUNT));
		sessionCashDrawer.setDrawerBleedAmount(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	private void calculateCashBack() {
		Criteria criteria = getCriteriaForTransaction(RefundTransaction.class);
		criteria.add(Restrictions.eq(PosTransaction.PROP_PAYMENT_TYPE_STRING, PaymentType.CASH.name()));
		criteria.setProjection(Projections.sum(RefundTransaction.PROP_AMOUNT));
		sessionCashDrawer.setCashBack(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	private void calculateMemberPayment() {
		Criteria criteria = getCriteriaForTransaction(CustomerAccountTransaction.class);
		criteria.add(Restrictions.eq(PosTransaction.PROP_PAYMENT_TYPE_STRING, PaymentType.MEMBER_ACCOUNT.name()));
		criteria.add(Restrictions.eq(PosTransaction.PROP_VOIDED, Boolean.FALSE));
		criteria.setProjection(Projections.sum(PosTransaction.PROP_AMOUNT)); //$NON-NLS-1$
		sessionCashDrawer.setCustomerPaymentAmount(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	private void calculateDrawerAccountable() {
		double drawerAccountableAmnt = sessionCashDrawer.getCashReceiptAmount() - sessionCashDrawer.getTipsPaid() - sessionCashDrawer.getPayOutAmount()
				- sessionCashDrawer.getCashBack() + sessionCashDrawer.getBeginCash() - sessionCashDrawer.getDrawerBleedAmount();
		sessionCashDrawer.setDrawerAccountable((drawerAccountableAmnt));
	}

	private void calculateTipsDiff() {
		double tipsDiff = sessionCashDrawer.getCashTips() + sessionCashDrawer.getChargedTips() - sessionCashDrawer.getTipsPaid();
		sessionCashDrawer.setTipsDifferential((tipsDiff));
	}

	private void calculateDeclaredTips() {
		Criteria criteria = session.createCriteria(DeclaredTips.class);
		criteria.add(Restrictions.eq(DeclaredTips.PROP_SESSION_ID, storeSession.getId()));
		criteria.setProjection(Projections.sum(CashDropTransaction.PROP_AMOUNT));
		sessionCashDrawer.setDeclaredTips(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	private void calculateToleranceAmount() {
		Criteria criteria = getCriteriaForTransaction(CashTransaction.class);
		criteria.add(Restrictions.eq(PosTransaction.PROP_STORE_SESSION_ID, storeSession.getId()));
		criteria.add(Restrictions.eq(PosTransaction.PROP_VOIDED, Boolean.FALSE));
		criteria.setProjection(Projections.sum(PosTransaction.PROP_TOLERANCE_AMOUNT));
		sessionCashDrawer.setToleranceAmount(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	public static CashDrawer createSessionSummaryReport(StoreSession currentSession) throws Exception {
		StoreSessionSummaryReportService reportService = new StoreSessionSummaryReportService(currentSession);
		CashDrawer sessionSummaryReport = reportService.populateStoreSessionSummaryReport();

		return sessionSummaryReport;
	}

}
