package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.beanutils.PropertyUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Order;

import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.model.Course;
import com.floreantpos.model.MenuItem;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class CourseDAO extends BaseCourseDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public CourseDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	@Override
	protected void delete(Object obj, Session session) {
		Course bean = (Course) obj;
		if (bean == null) {
			throw new PosException(Messages.getString("CourseDAO.0")); //$NON-NLS-1$
		}
		checkForeignRelation(bean);
		bean.setDeleted(Boolean.TRUE);
		update(bean, session);
	}

	private void checkForeignRelation(Course bean) throws PosException {
		try (Session session = createNewSession()) {
			List<String> foreignItemNames = GenericDAO.getInstance().getForeignDataListNames(session, MenuItem.class, MenuItem.PROP_COURSE_ID, bean.getId());
			if (foreignItemNames != null && foreignItemNames.size() > 0) {
				String details = Messages.getString("CourseDAO.1") + (foreignItemNames.size() > 1 ? Messages.getString("CourseDAO.2") : ":"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
				int count = 1;
				for (String itemName : foreignItemNames) {
					details += "\n" + count + ". " + itemName; //$NON-NLS-1$ //$NON-NLS-2$
					count++;
				}
				throw new PosException(Messages.getString("CourseDAO.6"), details); //$NON-NLS-1$
			}
		}
	}

	public List<Course> findAll() {
		Session session = null;
		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.addOrder(Order.asc(Course.PROP_SORT_ORDER));
			return criteria.list();
		} finally {
			closeSession(session);
		}
	}

	public void saveOrUpdateCourseList(List<Course> dataList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		if (dataList == null)
			return;

		Transaction tx = null;
		Session session = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();

			for (Iterator<Course> iterator = dataList.iterator(); iterator.hasNext();) {
				Course item = (Course) iterator.next();
				Course existingItem = get(item.getId());
				if (existingItem != null) {
					if (!BaseDataServiceDao.get().shouldSave(item.getLastUpdateTime(), existingItem.getLastUpdateTime())) {
						PosLog.info(getClass(), item.getName() + " already updated"); //$NON-NLS-1$
						continue;
					}
					//					long version = existingItem.getVersion();
					PropertyUtils.copyProperties(existingItem, item);
					//					existingItem.setVersion(version);
					existingItem.setUpdateLastUpdateTime(updateLastUpdateTime);
					existingItem.setUpdateSyncTime(updateSyncTime);
					update(existingItem, session);
				}
				else {
					item.setUpdateLastUpdateTime(updateLastUpdateTime);
					item.setUpdateSyncTime(updateSyncTime);
					save(item, session);
				}
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			closeSession(session);
		}
	}

}